// Variables globales para paginación
let currentPage = 1;
const itemsPerPage = 30;
let allTransactions = [];
let filteredTransactions = [];

// Actualizar hora en tiempo real
function updateTime() {
    const now = new Date();
    const timeString = now.toLocaleTimeString('es-ES', {
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    });
    const dateString = now.toLocaleDateString('es-ES', {
        weekday: 'short',
        day: 'numeric',
        month: 'short'
    });
    document.getElementById('current-time').innerHTML = `${dateString}<br>${timeString}`;
}

// Toggle filtros en móvil
function toggleFilters() {
    const filterSection = document.getElementById('filter-section');
    const isCollapsed = filterSection.classList.contains('collapsed');
    
    if (isCollapsed) {
        filterSection.classList.remove('collapsed');
    } else {
        filterSection.classList.add('collapsed');
    }
}

// FUNCIÓN ACTUALIZADA para renderizar transacciones en vista móvil CON CÓDIGO
function renderMobileTransactions(transactions) {
    const container = document.getElementById('mobile-transactions');
    container.innerHTML = '';

    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageTransactions = transactions.slice(startIndex, endIndex);

    pageTransactions.forEach((transaction, index) => {
        const card = document.createElement('div');
        card.className = 'transaction-card fade-in';
        
        // ACTUALIZADO: Incluir código de seguridad en la vista móvil
        let cardContent = `
            <div class="transaction-header">
                <div class="transaction-name">${transaction.numero ? `#${transaction.numero}` : `#${startIndex + index + 1}`} ${transaction.nombre}</div>
                <div class="transaction-amount">${transaction.monto}</div>
            </div>
            <div class="transaction-details">
                <div class="transaction-payment">${transaction.medioPago}</div>
                <div class="transaction-date">${transaction.fecha}</div>
            </div>
        `;
        
        // NUEVO: Agregar código de seguridad si existe
        if (transaction.codigo && transaction.codigo.trim() !== '') {
            cardContent += `
                <div class="transaction-security">
                    <small class="text-muted">
                        <i class="fas fa-key me-1"></i>
                        Código: ${transaction.codigo}
                    </small>
                </div>
            `;
        }
        
        card.innerHTML = cardContent;
        container.appendChild(card);
    });

    if (pageTransactions.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-inbox"></i>
                <h4>No hay transacciones</h4>
                <p>No se encontraron transacciones.</p>
            </div>
        `;
    }
}

// FUNCIÓN ACTUALIZADA para renderizar tabla desktop con paginación Y CÓDIGO
function renderDesktopTable(transactions) {
    // En desktop, aplicamos paginación a las filas existentes
    const allRows = document.querySelectorAll('#tabla-datos tr');
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    
    allRows.forEach((row, index) => {
        const isInCurrentPage = index >= startIndex && index < endIndex;
        const matchesFilter = transactions.some(t => {
            const cells = row.querySelectorAll('td');
            return cells.length >= 6 && // ACTUALIZADO: Ahora son 6 columnas (con código)
                   cells[1].textContent.trim() === t.nombre &&
                   cells[3].textContent.trim() === t.monto;
        });
        
        if (isInCurrentPage && matchesFilter) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });

    // Mostrar/ocultar estado vacío y tabla
    const emptyState = document.getElementById('empty-state');
    const table = document.querySelector('.table');
    
    if (transactions.length === 0) {
        table.style.display = 'none';
        emptyState.style.display = 'block';
    } else {
        table.style.display = 'table';
        emptyState.style.display = 'none';
    }
}

// Función para generar controles de paginación (SIN CAMBIOS)
function renderPagination(totalItems) {
    const totalPages = Math.ceil(totalItems / itemsPerPage);
    const container = document.getElementById('pagination-controls');
    
    container.innerHTML = '';

    // Info de paginación
    const startItem = (currentPage - 1) * itemsPerPage + 1;
    const endItem = Math.min(currentPage * itemsPerPage, totalItems);
    
    document.getElementById('pagination-showing').textContent = totalItems > 0 ? `${startItem}-${endItem}` : '0';
    document.getElementById('pagination-total').textContent = totalItems;

    if (totalPages <= 1) return;

    // Botón anterior
    const prevBtn = document.createElement('button');
    prevBtn.className = 'pagination-btn';
    prevBtn.innerHTML = '<i class="fas fa-chevron-left"></i>';
    prevBtn.disabled = currentPage === 1;
    prevBtn.onclick = () => {
        if (currentPage > 1) {
            currentPage--;
            updatePaginationDisplay();
        }
    };
    container.appendChild(prevBtn);

    // Botones de página (adaptado para móvil)
    const isMobile = window.innerWidth < 768;
    let startPage, endPage;

    if (isMobile) {
        // En móvil, mostrar solo página actual y adyacentes
        startPage = Math.max(1, currentPage - 1);
        endPage = Math.min(totalPages, currentPage + 1);
    } else {
        // En desktop, mostrar más páginas
        const maxVisible = 5;
        startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
        endPage = Math.min(totalPages, startPage + maxVisible - 1);
        
        if (endPage - startPage < maxVisible - 1) {
            startPage = Math.max(1, endPage - maxVisible + 1);
        }
    }

    // Primera página si no está visible
    if (startPage > 1) {
        const firstBtn = document.createElement('button');
        firstBtn.className = 'pagination-btn page-number';
        firstBtn.textContent = '1';
        firstBtn.onclick = () => {
            currentPage = 1;
            updatePaginationDisplay();
        };
        container.appendChild(firstBtn);

        if (startPage > 2) {
            const dots = document.createElement('span');
            dots.textContent = '...';
            dots.className = 'pagination-btn';
            dots.style.border = 'none';
            container.appendChild(dots);
        }
    }

    // Páginas visibles
    for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.className = `pagination-btn page-number ${i === currentPage ? 'active' : ''}`;
        pageBtn.textContent = i;
        pageBtn.onclick = () => {
            currentPage = i;
            updatePaginationDisplay();
        };
        container.appendChild(pageBtn);
    }

    // Última página si no está visible
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const dots = document.createElement('span');
            dots.textContent = '...';
            dots.className = 'pagination-btn';
            dots.style.border = 'none';
            container.appendChild(dots);
        }

        const lastBtn = document.createElement('button');
        lastBtn.className = 'pagination-btn page-number';
        lastBtn.textContent = totalPages;
        lastBtn.onclick = () => {
            currentPage = totalPages;
            updatePaginationDisplay();
        };
        container.appendChild(lastBtn);
    }

    // Botón siguiente
    const nextBtn = document.createElement('button');
    nextBtn.className = 'pagination-btn';
    nextBtn.innerHTML = '<i class="fas fa-chevron-right"></i>';
    nextBtn.disabled = currentPage === totalPages;
    nextBtn.onclick = () => {
        if (currentPage < totalPages) {
            currentPage++;
            updatePaginationDisplay();
        }
    };
    container.appendChild(nextBtn);
}

// Función para actualizar solo la visualización de paginación (NO interfiere con scripts.js)
function updatePaginationDisplay() {
    renderMobileTransactions(filteredTransactions);
    renderDesktopTable(filteredTransactions);
    renderPagination(filteredTransactions.length);
    updateStatsFromTable(); // Usar función diferente para estadísticas
}

// Función para actualizar estadísticas (NO interfiere con la lógica original)
function updateStatsFromTable() {
    const rows = document.querySelectorAll('#tabla-datos tr:not(.d-none)');
    let totalAmount = 0;
    let todayCount = 0;
    const uniqueUsers = new Set();
    const today = new Date().toISOString().split('T')[0];
    const todayFormatted = new Date().toLocaleDateString('es-PE');
    
    rows.forEach((row, index) => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 6) { // ACTUALIZADO: Ahora son 6 columnas
            const name = cells[1].textContent.trim();
            const montoText = cells[3].textContent.trim();
            let amount = 0;
            
            if (montoText) {
                const cleanAmount = montoText.replace(/[S\/\s,]/g, '');
                amount = parseFloat(cleanAmount) || 0;
            }
            
            const dateText = cells[5].textContent.trim(); // ACTUALIZADO: La fecha está en la columna 5 ahora
            
            if (name) {
                uniqueUsers.add(name);
            }
            
            totalAmount += amount;
            
            if (dateText && (
                dateText.includes(today) || 
                dateText.includes(todayFormatted) ||
                dateText.includes('2025-07-26') ||
                dateText.includes('26/07/2025') ||
                dateText.includes('Hoy')
            )) {
                todayCount++;
            }
        }
    });

    // Actualizar elementos
    const updateElement = (id, value) => {
        const el = document.getElementById(id);
        if (el) el.textContent = value;
    };

    const formattedAmount = `S/ ${totalAmount.toLocaleString('es-PE', {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
    
    updateElement('total-amount', formattedAmount);
    updateElement('total-amount-mobile', formattedAmount);
    updateElement('total-transactions', filteredTransactions.length);
    updateElement('total-transactions-mobile', filteredTransactions.length);
    updateElement('today-transactions', todayCount);
    updateElement('today-transactions-mobile', todayCount);
    updateElement('unique-users', uniqueUsers.size);
    updateElement('unique-users-mobile', uniqueUsers.size);
    updateElement('resultados-count', filteredTransactions.length);
}

// FUNCIÓN ACTUALIZADA para extraer datos de la tabla HTML CON CÓDIGO DE SEGURIDAD
function extractDataFromTable() {
    const rows = document.querySelectorAll('#tabla-datos tr');
    allTransactions = [];
    
    rows.forEach(row => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 6) { // ACTUALIZADO: Ahora son 6 columnas
            allTransactions.push({
                numero: cells[0].textContent.trim(),
                nombre: cells[1].textContent.trim(),
                medioPago: cells[2].textContent.trim(),
                monto: cells[3].textContent.trim(),
                codigo: cells[4].textContent.trim(), // NUEVO: Código de seguridad
                fecha: cells[5].textContent.trim() // ACTUALIZADO: La fecha está en la columna 5
            });
        }
    });
    
    filteredTransactions = [...allTransactions];
}

// FUNCIÓN ACTUALIZADA para aplicar filtros CON CÓDIGO DE SEGURIDAD
function aplicarFiltroPaginado() {
    const nombre = document.getElementById('nombre').value.toLowerCase();
    const fecha = document.getElementById('fecha').value;
    const montoMin = parseFloat(document.getElementById('monto-min').value) || 0;
    const codigoSeguridad = document.getElementById('codigo-seguridad') ? document.getElementById('codigo-seguridad').value.toLowerCase() : '';
    
    // Primero aplicar filtro usando el método original (scripts.js)
    if (typeof aplicarFiltro === 'function') {
        aplicarFiltro(); // Llamar a la función original de scripts.js
    }
    
    // Luego extraer datos actualizados y aplicar paginación
    setTimeout(() => {
        extractDataFromTable();
        
        // ACTUALIZADO: Aplicar filtros locales para la vista móvil CON CÓDIGO
        filteredTransactions = allTransactions.filter(transaction => {
            let mostrar = true;
            
            if (nombre && !transaction.nombre.toLowerCase().includes(nombre)) {
                mostrar = false;
            }
            
            if (fecha && !transaction.fecha.includes(fecha)) {
                mostrar = false;
            }
            
            const monto = parseFloat(transaction.monto.replace(/[S\/\s,]/g, ''));
            if (monto < montoMin) {
                mostrar = false;
            }
            
            // NUEVO: Filtro por código de seguridad
            if (codigoSeguridad && !transaction.codigo.toLowerCase().includes(codigoSeguridad)) {
                mostrar = false;
            }
            
            return mostrar;
        });

        currentPage = 1; // Reset a primera página
        updatePaginationDisplay();
    }, 200); // Esperar a que scripts.js actualice la tabla
}

// FUNCIÓN ACTUALIZADA para limpiar filtros CON CÓDIGO
function limpiarFiltrosPaginado() {
    document.getElementById('nombre').value = '';
    document.getElementById('fecha').value = '';
    document.getElementById('monto-min').value = '';
    const codigoInput = document.getElementById('codigo-seguridad');
    if (codigoInput) {
        codigoInput.value = '';
    }
    aplicarFiltroPaginado();
}

// Hook para detectar cuando scripts.js actualiza la tabla (SIN CAMBIOS)
function hookTableUpdates() {
    const tablaDatos = document.getElementById('tabla-datos');
    if (tablaDatos) {
        const observer = new MutationObserver(() => {
            // Cuando la tabla se actualiza, extraer datos y actualizar paginación
            setTimeout(() => {
                extractDataFromTable();
                updatePaginationDisplay();
            }, 100);
        });
        
        observer.observe(tablaDatos, {
            childList: true,
            subtree: true
        });
    }
}

// FUNCIÓN ACTUALIZADA para inicializar sistema CON NUEVO CAMPO DE CÓDIGO
document.addEventListener('DOMContentLoaded', function() {
    console.log('Inicializando sistema de paginación...');
    
    updateTime();
    setInterval(updateTime, 1000);
    
    // Hook para detectar actualizaciones de la tabla
    hookTableUpdates();
    
    // Extraer datos iniciales después de que scripts.js cargue
    setTimeout(() => {
        extractDataFromTable();
        updatePaginationDisplay();
    }, 1000);
    
    // Manejar cambios de orientación/tamaño
    window.addEventListener('resize', () => {
        setTimeout(updatePaginationDisplay, 100);
    });
    
    // ACTUALIZADO: Reemplazar las funciones de filtro INCLUYENDO EL NUEVO CAMPO
    const nombreInput = document.getElementById('nombre');
    const fechaInput = document.getElementById('fecha');
    const montoInput = document.getElementById('monto-min');
    const codigoInput = document.getElementById('codigo-seguridad'); // NUEVO
    
    if (nombreInput) {
        nombreInput.removeAttribute('oninput');
        nombreInput.addEventListener('input', aplicarFiltroPaginado);
    }
    
    if (fechaInput) {
        fechaInput.removeAttribute('oninput');
        fechaInput.addEventListener('input', aplicarFiltroPaginado);
    }
    
    if (montoInput) {
        montoInput.removeAttribute('oninput');
        montoInput.addEventListener('input', aplicarFiltroPaginado);
    }
    
    // NUEVO: Event listener para el campo de código de seguridad
    if (codigoInput) {
        codigoInput.addEventListener('input', aplicarFiltroPaginado);
    }
});

// Variables para auto-actualización (diferentes a scripts.js) - SIN CAMBIOS
let paginationAutoUpdateInterval;
let isPaginationAutoUpdateEnabled = true;

// Función para pausar/reanudar auto-actualización (sin interferir con scripts.js) - SIN CAMBIOS
function toggleAutoUpdate() {
    isPaginationAutoUpdateEnabled = !isPaginationAutoUpdateEnabled;
    const btn = document.getElementById('toggle-auto-update');
    
    if (isPaginationAutoUpdateEnabled) {
        btn.innerHTML = '<i class="fas fa-pause"></i> <span class="d-none d-md-inline">Auto-update</span>';
        btn.className = 'action-btn btn-warning';
    } else {
        btn.innerHTML = '<i class="fas fa-play"></i> <span class="d-none d-md-inline">Reanudar</span>';
        btn.className = 'action-btn btn-success';
    }
}

// Función para actualizar manualmente (SIN interferir con scripts.js) - SIN CAMBIOS
function cargarDatosDesdeServidor() {
    // Simplemente re-extraer datos de la tabla actual
    extractDataFromTable();
    updatePaginationDisplay();
    console.log('Datos de paginación actualizados');
}

// Función para limpiar transacciones (usar la del scripts.js original) - SIN CAMBIOS
function limpiarTodasLasTransacciones() {
    if (typeof window.limpiarTodasLasTransacciones === 'function') {
        window.limpiarTodasLasTransacciones();
    } else if (confirm('¿Eliminar todas las transacciones?')) {
        // Fallback si no existe la función original
        fetch('./limpiar_todas.php')
            .then(response => response.text())
            .then(data => {
                alert(data);
                // La tabla se actualizará automáticamente por el hook
            })
            .catch(error => console.error('Error:', error));
    }
}