// ===== VARIABLES GLOBALES =====
let ultimoId = 0;
let sonidoPermitido = false;
let audioContext = null;

// Variables para paginación
let currentPage = 1;
const itemsPerPage = 30;
let allTransactions = [];
let filteredTransactions = [];

// Variables para auto-actualización
let paginationAutoUpdateInterval;
let isPaginationAutoUpdateEnabled = true;

// ===== FUNCIONES DE AUDIO =====
function inicializarAudio() {
    if (!audioContext) {
        audioContext = new (window.AudioContext || window.webkitAudioContext)();
    }
    sonidoPermitido = true;
}

function reproducirSonido() {
    if (sonidoPermitido && audioContext) {
        try {
            const sonido = new Audio('./sonidos/notificacion.mp3');
            sonido.volume = 0.7;
            sonido.play().catch(() => {});
        } catch (error) {}
    }
}

// ===== FUNCIONES DE NOTIFICACIONES =====
function mostrarNotificacionSilenciosa(titulo, mensaje) {
    if ("Notification" in window && Notification.permission === "granted") {
        new Notification(titulo, {
            body: mensaje,
            icon: "https://quipusfact.site/fibranet/logo_fibra.png",
            silent: true,
            requireInteraction: false
        });
    }
}

// ===== FUNCIONES DE TABLA =====
function actualizarTabla() {
    fetch('./obtener_datos.php')
        .then(response => response.text())
        .then(data => {
            document.getElementById('tabla-datos').innerHTML = data;
            extractDataFromTable();
            updatePaginationDisplay();
        })
        .catch(() => {});
}

// FUNCIÓN OPTIMIZADA para verificar nuevos registros
function verificarNuevoRegistro() {
    fetch('./obtener_ultimo_id.php')
        .then(response => response.json())
        .then(data => {
            if (data.id > ultimoId) {
                const titulo = "¡Nuevo Yape!";
                let mensaje = `Nombre: ${data.transaction_name}\nMonto: ${data.amount}\nFecha: ${data.date_time}`;
                
                if (data.security_code) {
                    mensaje += `\nCódigo: ${data.security_code}`;
                }

                mostrarNotificacionSilenciosa(titulo, mensaje);
                ultimoId = data.id;
                actualizarTabla();
                reproducirSonido();
            }
        })
        .catch(() => {});
}

// ===== FUNCIONES DE FILTROS =====
function aplicarFiltro() {
    const nombre = document.getElementById('nombre').value;
    const fecha = document.getElementById('fecha').value;
    const montoMin = document.getElementById('monto-min').value;
    const codigoSeguridad = document.getElementById('codigo-seguridad') ? document.getElementById('codigo-seguridad').value : '';

    const url = `./obtener_datos.php?nombre=${encodeURIComponent(nombre)}&fecha=${encodeURIComponent(fecha)}&monto_min=${encodeURIComponent(montoMin)}&codigo_seguridad=${encodeURIComponent(codigoSeguridad)}`;

    fetch(url)
        .then(response => response.text())
        .then(data => {
            document.getElementById('tabla-datos').innerHTML = data;
            extractDataFromTable();
            currentPage = 1;
            updatePaginationDisplay();
        })
        .catch(() => {});
}

function limpiarFiltros() {
    document.getElementById('nombre').value = '';
    document.getElementById('fecha').value = '';
    document.getElementById('monto-min').value = '';
    const codigoInput = document.getElementById('codigo-seguridad');
    if (codigoInput) codigoInput.value = '';
    aplicarFiltro();
}

// ===== FUNCIONES DE PAGINACIÓN =====
function updateTime() {
    const now = new Date();
    const timeString = now.toLocaleTimeString('es-ES', {
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
    });
    const dateString = now.toLocaleDateString('es-ES', {
        weekday: 'short',
        day: 'numeric',
        month: 'short'
    });
    const currentTimeEl = document.getElementById('current-time');
    if (currentTimeEl) {
        currentTimeEl.innerHTML = `${dateString}<br>${timeString}`;
    }
}

function toggleFilters() {
    const filterSection = document.getElementById('filter-section');
    filterSection.classList.toggle('collapsed');
}

function renderMobileTransactions(transactions) {
    const container = document.getElementById('mobile-transactions');
    if (!container) return;
    
    container.innerHTML = '';

    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageTransactions = transactions.slice(startIndex, endIndex);

    pageTransactions.forEach((transaction, index) => {
        const card = document.createElement('div');
        card.className = 'transaction-card fade-in';
        
        let cardContent = `
            <div class="transaction-header">
                <div class="transaction-name">${transaction.numero ? `#${transaction.numero}` : `#${startIndex + index + 1}`} ${transaction.nombre}</div>
                <div class="transaction-amount">${transaction.monto}</div>
            </div>
            <div class="transaction-details">
                <div class="transaction-payment">${transaction.medioPago}</div>
                <div class="transaction-date">${transaction.fecha}</div>
            </div>
        `;
        
        if (transaction.codigo && transaction.codigo.trim() !== '' && transaction.codigo !== '-') {
            cardContent += `
                <div class="transaction-security">
                    <small class="text-muted">
                        <i class="fas fa-key me-1"></i>
                        Código: ${transaction.codigo}
                    </small>
                </div>
            `;
        }
        
        card.innerHTML = cardContent;
        container.appendChild(card);
    });

    if (pageTransactions.length === 0) {
        container.innerHTML = `
            <div class="empty-state">
                <i class="fas fa-inbox"></i>
                <h4>No hay transacciones</h4>
                <p>No se encontraron transacciones.</p>
            </div>
        `;
    }
}

function renderDesktopTable(transactions) {
    const allRows = document.querySelectorAll('#tabla-datos tr');
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    
    allRows.forEach((row, index) => {
        const isInCurrentPage = index >= startIndex && index < endIndex;
        const matchesFilter = transactions.some(t => {
            const cells = row.querySelectorAll('td');
            return cells.length >= 6 &&
                   cells[1].textContent.trim() === t.nombre &&
                   cells[3].textContent.trim() === t.monto;
        });
        
        row.style.display = (isInCurrentPage && matchesFilter) ? '' : 'none';
    });

    const emptyState = document.getElementById('empty-state');
    const table = document.querySelector('.table');
    
    if (transactions.length === 0 && emptyState && table) {
        table.style.display = 'none';
        emptyState.style.display = 'block';
    } else if (emptyState && table) {
        table.style.display = 'table';
        emptyState.style.display = 'none';
    }
}

function renderPagination(totalItems) {
    const totalPages = Math.ceil(totalItems / itemsPerPage);
    const container = document.getElementById('pagination-controls');
    if (!container) return;
    
    container.innerHTML = '';

    const startItem = (currentPage - 1) * itemsPerPage + 1;
    const endItem = Math.min(currentPage * itemsPerPage, totalItems);
    
    const showingEl = document.getElementById('pagination-showing');
    const totalEl = document.getElementById('pagination-total');
    
    if (showingEl) showingEl.textContent = totalItems > 0 ? `${startItem}-${endItem}` : '0';
    if (totalEl) totalEl.textContent = totalItems;

    if (totalPages <= 1) return;

    // Botón anterior
    const prevBtn = document.createElement('button');
    prevBtn.className = 'pagination-btn';
    prevBtn.innerHTML = '<i class="fas fa-chevron-left"></i>';
    prevBtn.disabled = currentPage === 1;
    prevBtn.onclick = () => {
        if (currentPage > 1) {
            currentPage--;
            updatePaginationDisplay();
        }
    };
    container.appendChild(prevBtn);

    // Botones de página
    const isMobile = window.innerWidth < 768;
    let startPage, endPage;

    if (isMobile) {
        startPage = Math.max(1, currentPage - 1);
        endPage = Math.min(totalPages, currentPage + 1);
    } else {
        const maxVisible = 5;
        startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
        endPage = Math.min(totalPages, startPage + maxVisible - 1);
        
        if (endPage - startPage < maxVisible - 1) {
            startPage = Math.max(1, endPage - maxVisible + 1);
        }
    }

    // Primera página si no está visible
    if (startPage > 1) {
        const firstBtn = document.createElement('button');
        firstBtn.className = 'pagination-btn page-number';
        firstBtn.textContent = '1';
        firstBtn.onclick = () => {
            currentPage = 1;
            updatePaginationDisplay();
        };
        container.appendChild(firstBtn);

        if (startPage > 2) {
            const dots = document.createElement('span');
            dots.textContent = '...';
            dots.className = 'pagination-btn';
            dots.style.border = 'none';
            container.appendChild(dots);
        }
    }

    // Páginas visibles
    for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.className = `pagination-btn page-number ${i === currentPage ? 'active' : ''}`;
        pageBtn.textContent = i;
        pageBtn.onclick = () => {
            currentPage = i;
            updatePaginationDisplay();
        };
        container.appendChild(pageBtn);
    }

    // Última página si no está visible
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const dots = document.createElement('span');
            dots.textContent = '...';
            dots.className = 'pagination-btn';
            dots.style.border = 'none';
            container.appendChild(dots);
        }

        const lastBtn = document.createElement('button');
        lastBtn.className = 'pagination-btn page-number';
        lastBtn.textContent = totalPages;
        lastBtn.onclick = () => {
            currentPage = totalPages;
            updatePaginationDisplay();
        };
        container.appendChild(lastBtn);
    }

    // Botón siguiente
    const nextBtn = document.createElement('button');
    nextBtn.className = 'pagination-btn';
    nextBtn.innerHTML = '<i class="fas fa-chevron-right"></i>';
    nextBtn.disabled = currentPage === totalPages;
    nextBtn.onclick = () => {
        if (currentPage < totalPages) {
            currentPage++;
            updatePaginationDisplay();
        }
    };
    container.appendChild(nextBtn);
}

function updatePaginationDisplay() {
    renderMobileTransactions(filteredTransactions);
    renderDesktopTable(filteredTransactions);
    renderPagination(filteredTransactions.length);
    updateStatsFromTable();
}

function updateStatsFromTable() {
    const rows = document.querySelectorAll('#tabla-datos tr:not(.d-none)');
    let totalAmount = 0;
    let todayCount = 0;
    const uniqueUsers = new Set();
    const today = new Date().toISOString().split('T')[0];
    const todayFormatted = new Date().toLocaleDateString('es-PE');
    
    rows.forEach((row) => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 6) {
            const name = cells[1].textContent.trim();
            const montoText = cells[3].textContent.trim();
            let amount = 0;
            
            if (montoText) {
                const cleanAmount = montoText.replace(/[S\/\s,]/g, '');
                amount = parseFloat(cleanAmount) || 0;
            }
            
            const dateText = cells[5].textContent.trim();
            
            if (name) {
                uniqueUsers.add(name);
            }
            
            totalAmount += amount;
            
            if (dateText && (
                dateText.includes(today) || 
                dateText.includes(todayFormatted) ||
                dateText.includes('2025-07-26') ||
                dateText.includes('26/07/2025') ||
                dateText.includes('Hoy')
            )) {
                todayCount++;
            }
        }
    });

    const updateElement = (id, value) => {
        const el = document.getElementById(id);
        if (el) el.textContent = value;
    };

    const formattedAmount = `S/ ${totalAmount.toLocaleString('es-PE', {minimumFractionDigits: 2, maximumFractionDigits: 2})}`;
    
    updateElement('total-amount', formattedAmount);
    updateElement('total-amount-mobile', formattedAmount);
    updateElement('total-transactions', filteredTransactions.length);
    updateElement('total-transactions-mobile', filteredTransactions.length);
    updateElement('today-transactions', todayCount);
    updateElement('today-transactions-mobile', todayCount);
    updateElement('unique-users', uniqueUsers.size);
    updateElement('unique-users-mobile', uniqueUsers.size);
    updateElement('resultados-count', filteredTransactions.length);
}

function extractDataFromTable() {
    const rows = document.querySelectorAll('#tabla-datos tr');
    allTransactions = [];
    
    rows.forEach(row => {
        const cells = row.querySelectorAll('td');
        if (cells.length >= 6) {
            allTransactions.push({
                numero: cells[0].textContent.trim(),
                nombre: cells[1].textContent.trim(),
                medioPago: cells[2].textContent.trim(),
                monto: cells[3].textContent.trim(),
                codigo: cells[4].textContent.trim(),
                fecha: cells[5].textContent.trim()
            });
        }
    });
    
    filteredTransactions = [...allTransactions];
}

// ===== FUNCIONES DE CONTROL =====
function toggleAutoUpdate() {
    isPaginationAutoUpdateEnabled = !isPaginationAutoUpdateEnabled;
    const btn = document.getElementById('toggle-auto-update');
    
    if (btn) {
        if (isPaginationAutoUpdateEnabled) {
            btn.innerHTML = '<i class="fas fa-pause"></i> <span class="d-none d-md-inline">Auto-update</span>';
            btn.className = 'action-btn btn-warning';
        } else {
            btn.innerHTML = '<i class="fas fa-play"></i> <span class="d-none d-md-inline">Reanudar</span>';
            btn.className = 'action-btn btn-success';
        }
    }
}

function cargarDatosDesdeServidor() {
    actualizarTabla();
}

function limpiarTodasLasTransacciones() {
    if (confirm("¿Estás seguro de que deseas eliminar todas las transacciones?")) {
        fetch('./limpiar_todas.php')
            .then(response => response.text())
            .then(data => {
                alert(data);
                actualizarTabla();
            })
            .catch(() => {});
    }
}

// ===== INICIALIZACIÓN OPTIMIZADA =====
document.addEventListener('DOMContentLoaded', function() {
    // Inicializar reloj
    updateTime();
    setInterval(updateTime, 1000);
    
    // Configurar botón de sonido
    const botonSonido = document.getElementById('boton-sonido');
    if (botonSonido) {
        botonSonido.addEventListener('click', () => {
            inicializarAudio();
            
            if (botonSonido.classList.contains('active')) {
                botonSonido.classList.remove('active');
                botonSonido.innerHTML = '<i class="fas fa-volume-up"></i> Sonido';
                botonSonido.style.background = 'linear-gradient(135deg, var(--primary-color), var(--secondary-color))';
                sonidoPermitido = false;
            } else {
                botonSonido.classList.add('active');
                botonSonido.innerHTML = '<i class="fas fa-volume-off"></i> Mute';
                botonSonido.style.background = 'linear-gradient(135deg, var(--success-color), #059669)';
                sonidoPermitido = true;
                reproducirSonido();
            }
        });
    }
    
    // Configurar event listeners para filtros
    const nombreInput = document.getElementById('nombre');
    const fechaInput = document.getElementById('fecha');
    const montoInput = document.getElementById('monto-min');
    const codigoInput = document.getElementById('codigo-seguridad');
    
    if (nombreInput) {
        nombreInput.removeAttribute('oninput');
        nombreInput.addEventListener('input', aplicarFiltro);
    }
    
    if (fechaInput) {
        fechaInput.removeAttribute('oninput');
        fechaInput.addEventListener('input', aplicarFiltro);
    }
    
    if (montoInput) {
        montoInput.removeAttribute('oninput');
        montoInput.addEventListener('input', aplicarFiltro);
    }
    
    if (codigoInput) {
        codigoInput.addEventListener('input', aplicarFiltro);
    }
    
    // Observer para cambios en la tabla
    const tablaDatos = document.getElementById('tabla-datos');
    if (tablaDatos) {
        const observer = new MutationObserver(() => {
            extractDataFromTable();
            updatePaginationDisplay();
        });
        
        observer.observe(tablaDatos, {
            childList: true,
            subtree: true
        });
    }
    
    // 🚀 CARGA INMEDIATA (sin timeout)
    actualizarTabla();
    
    // Manejar cambios de orientación/tamaño
    window.addEventListener('resize', () => {
        updatePaginationDisplay();
    });
    
    // Activación automática de sonido
    let primeraInteraccion = true;
    
    function manejarPrimeraInteraccion() {
        if (primeraInteraccion) {
            inicializarAudio();
            primeraInteraccion = false;
            
            if (botonSonido && !botonSonido.classList.contains('active')) {
                botonSonido.click();
            }
            
            document.removeEventListener('click', manejarPrimeraInteraccion);
            document.removeEventListener('keydown', manejarPrimeraInteraccion);
            document.removeEventListener('touchstart', manejarPrimeraInteraccion);
        }
    }
    
    document.addEventListener('click', manejarPrimeraInteraccion);
    document.addEventListener('keydown', manejarPrimeraInteraccion);
    document.addEventListener('touchstart', manejarPrimeraInteraccion);
});

// Pedir permiso para notificaciones
if (Notification.permission !== "granted" && Notification.permission !== "denied") {
    Notification.requestPermission();
}

// Ejecutar verificación cada 2 segundos
setInterval(verificarNuevoRegistro, 2000);