<?php
header('Content-Type: application/json');

include 'config/bd.php';

// Archivo de log para depuración
$logFile = 'log.txt';

// Constante para el nombre de la empresa
define('EMPRESA_NAME', 'INTERSAT DEL VALLE SAC');

// Respuesta inicial
$response = ["success" => false, "message" => ""];

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Establecer la zona horaria a Lima, Perú
    date_default_timezone_set('America/Lima');

    // Verificar si es una verificación de conexión
    if (isset($_POST['action']) && $_POST['action'] === 'check_connection') {
        $response["success"] = true;
        $response["message"] = "Conexión exitosa al webservice.";
        
        // Registrar verificación de conexión
        file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Verificación de conexión exitosa" . PHP_EOL, FILE_APPEND);
    }
    // Verificar que los datos de transacción fueron enviados
    elseif (isset($_POST['transaction_name'], $_POST['package_name'], $_POST['amount'])) {
        $transactionName = trim($_POST['transaction_name']);
        $packageName = trim($_POST['package_name']);
        $amount = floatval($_POST['amount']);
        $securityCode = isset($_POST['security_code']) ? trim($_POST['security_code']) : null;
        $empresa = EMPRESA_NAME; // Constante de empresa
        $dateTime = date("Y-m-d H:i:s"); // Fecha y hora actual

        // Validar datos básicos
        if (empty($transactionName) || empty($packageName) || $amount <= 0) {
            $response["message"] = "Datos de transacción inválidos.";
            file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Error: Datos inválidos - " . json_encode($_POST) . PHP_EOL, FILE_APPEND);
        } else {
            // Verificar si ya existe una transacción similar reciente (últimos 5 minutos)
            $checkStmt = $pdo->prepare("
                SELECT COUNT(*) as count 
                FROM transactions 
                WHERE transaction_name = ? 
                AND amount = ? 
                AND security_code = ? 
                AND empresa = ?
                AND date_time > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
            ");
            $checkStmt->execute([$transactionName, $amount, $securityCode, $empresa]);
            $existingCount = $checkStmt->fetch(PDO::FETCH_ASSOC)['count'];

            if ($existingCount > 0) {
                $response["success"] = true;
                $response["message"] = "Transacción duplicada ignorada (ya existe una similar reciente).";
                file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Transacción duplicada ignorada: $transactionName - $amount - $securityCode - $empresa" . PHP_EOL, FILE_APPEND);
            } else {
                // Insertar datos en la base de datos incluyendo empresa
                $stmt = $pdo->prepare("
                    INSERT INTO transactions (transaction_name, package_name, amount, security_code, empresa, date_time) 
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $insertSuccess = $stmt->execute([$transactionName, $packageName, $amount, $securityCode, $empresa, $dateTime]);

                if ($insertSuccess) {
                    // Obtener el ID de la transacción insertada
                    $transactionId = $pdo->lastInsertId();

                    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] ✅ TRANSACCIÓN INSERTADA - ID: $transactionId - Empresa: $empresa" . PHP_EOL, FILE_APPEND);

                    // Preparar datos para la notificación
                    $notificationData = [
                        'transaction_id' => $transactionId,
                        'transaction_name' => $transactionName,
                        'amount' => $amount,
                        'security_code' => $securityCode,
                        'package_name' => $packageName,
                        'empresa' => $empresa,
                        'date_time' => $dateTime
                    ];

                    // USAR INCLUDE DIRECTO (MÁS EFICIENTE QUE cURL)
                    $notificationSuccess = false;
                    $notificationResponse = '';
                    
                    try {
                        file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] 🚀 INICIANDO INCLUDE DIRECTO de send_notification.php" . PHP_EOL, FILE_APPEND);
                        
                        // Backup del POST original
                        $_POST_backup = $_POST;
                        
                        // Establecer los datos para send_notification.php
                        $_POST = $notificationData;
                        
                        // Capturar la salida de send_notification.php
                        ob_start();
                        include 'send_notification.php';
                        $notificationResponse = ob_get_clean();
                        
                        // Restaurar POST original
                        $_POST = $_POST_backup;
                        
                        file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] 📥 Respuesta de send_notification: " . substr($notificationResponse, 0, 300) . PHP_EOL, FILE_APPEND);
                        
                        // Intentar parsear la respuesta JSON
                        $jsonResponse = json_decode($notificationResponse, true);
                        
                        if ($jsonResponse && isset($jsonResponse['success'])) {
                            $notificationSuccess = $jsonResponse['success'];
                            if ($jsonResponse['success']) {
                                file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] ✅ FIREBASE NOTIFICATION ENVIADA EXITOSAMENTE" . PHP_EOL, FILE_APPEND);
                            } else {
                                file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] ⚠️ Firebase reportó error: " . ($jsonResponse['message'] ?? 'Unknown') . PHP_EOL, FILE_APPEND);
                            }
                        } else {
                            // Si no es JSON válido, pero no hay errores PHP, asumir éxito
                            $notificationSuccess = (strpos($notificationResponse, 'Fatal error') === false && 
                                                  strpos($notificationResponse, 'Parse error') === false);
                            file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] 📋 Respuesta no es JSON válido, asumiendo " . ($notificationSuccess ? 'éxito' : 'error') . PHP_EOL, FILE_APPEND);
                        }
                        
                    } catch (Exception $e) {
                        file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] ❌ Error en INCLUDE: " . $e->getMessage() . PHP_EOL, FILE_APPEND);
                        $notificationSuccess = false;
                    }

                    // Respuesta exitosa
                    $response["success"] = true;
                    $response["message"] = "Transacción almacenada correctamente.";
                    $response["data"] = [
                        "transaction_id" => $transactionId,
                        "transaction_name" => $transactionName,
                        "package_name" => $packageName,
                        "amount" => $amount,
                        "security_code" => $securityCode,
                        "empresa" => $empresa,
                        "timestamp" => $dateTime,
                        "notification_sent" => $notificationSuccess,
                        "notification_response" => substr($notificationResponse, 0, 200)
                    ];

                    // Log final
                    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] 🏁 PROCESO COMPLETADO - ID=$transactionId, Empresa=$empresa, Notificación=" . ($notificationSuccess ? '✅ OK' : '❌ FAIL') . PHP_EOL, FILE_APPEND);
                } else {
                    $response["message"] = "Error al insertar la transacción en la base de datos.";
                    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] ❌ ERROR: Fallo en INSERT" . PHP_EOL, FILE_APPEND);
                }
            }
        }
    } else {
        $response["message"] = "Datos incompletos o acción no reconocida.";
        file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Error: Datos incompletos - " . json_encode($_POST) . PHP_EOL, FILE_APPEND);
    }

} catch (PDOException $e) {
    $response["message"] = "Error en la base de datos: " . $e->getMessage();
    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Error PDO: " . $e->getMessage() . PHP_EOL, FILE_APPEND);
    http_response_code(500);
} catch (Exception $e) {
    $response["message"] = "Error general: " . $e->getMessage();
    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Error general: " . $e->getMessage() . PHP_EOL, FILE_APPEND);
    http_response_code(500);
}

// Registrar todas las solicitudes (excepto verificaciones de conexión frecuentes)
if (!isset($_POST['action']) || $_POST['action'] !== 'check_connection') {
    file_put_contents($logFile, "[" . date("Y-m-d H:i:s") . "] Request: " . json_encode($_POST) . " | Response: " . json_encode($response) . PHP_EOL, FILE_APPEND);
}

echo json_encode($response);
?>